<?php
/**
 * Plugin Name: Mi Centro de Control WooCommerce Personalizado V2
 * Description: Personaliza el escritorio del backend de WordPress para un usuario específico con un centro de control de WooCommerce interactivo y oculta el panel de actividad. Ahora permite añadir múltiples usuarios desde los ajustes con una interfaz mejorada.
 * Version: 2.4
 * Author: Hernán De Vicente
 */

// Define el slug de la página de dashboard personalizado
define( 'CUSTOM_WOO_DASHBOARD_SLUG', 'custom-woo-dashboard' );
// Define el nombre de la opción en la base de datos para los usuarios afectados
define( 'WOO_CUSTOM_DASHBOARD_USERS_OPTION', 'woo_custom_dashboard_affected_users' );

// 1. Añadir páginas de menú personalizadas al panel de administración
function custom_woo_dashboard_menu_page() {
    // Página principal del centro de control
    add_menu_page(
        'Centro WooCommerce',          // Título de la página
        'Centro WooCommerce',          // Texto del menú
        'edit_products',               // Capacidad requerida: solo usuarios que puedan editar productos
        CUSTOM_WOO_DASHBOARD_SLUG,     // Slug del menú (identificador único)
        'display_custom_woo_dashboard_content', // Función que mostrará el contenido
        'dashicons-store',             // Icono de la tienda de Dashicons
        2                              // Posición en el menú (2 es justo debajo de "Escritorio")
    );

    // Subpágina para los ajustes de usuarios (solo visible para administradores)
    add_submenu_page(
        CUSTOM_WOO_DASHBOARD_SLUG,       // Parent slug
        'Ajustes de Usuarios WooCommerce', // Page title
        'Ajustes Usuarios',              // Menu title
        'manage_options',                // Capability required: only administrators
        'woo-dashboard-user-settings',   // Menu slug
        'display_woo_dashboard_user_settings' // Callback function
    );
}
add_action( 'admin_menu', 'custom_woo_dashboard_menu_page' );

// 2. Función para mostrar el contenido HTML personalizado de la página de WooCommerce
function display_custom_woo_dashboard_content() {
    ?>
    <div class="wrap custom-woo-dashboard-wrap">
        <h1 class="wp-heading-inline">Tu Centro de Control WooCommerce</h1>
        <p class="subtitle">Gestiona tu tienda de forma rápida y sencilla.</p>

        <div class="woo-cards-container">

            <div class="woo-card" data-card="pedidos">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-cart"></span> Pedidos</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Gestiona y procesa todos los pedidos de tu tienda.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('edit.php?post_type=shop_order') ); ?>" class="button button-primary">Ver Pedidos</a>
                    <a href="<?php echo esc_url( admin_url('admin.php?page=wc-settings&tab=checkout') ); ?>" class="button button-secondary">Ajustes de Pago</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button> <h3>Cómo Gestionar Pedidos</h3>
                    <p>En esta sección, podrás ver un listado de todos los pedidos recibidos. Cada pedido tiene un estado (Pendiente de pago, Procesando, Completado, etc.) que puedes actualizar manualmente.</p>
                    <ol>
                        <li>**Ver detalles:** Haz clic en el título de un pedido para ver los productos, datos del cliente, dirección de envío y facturación.</li>
                        <li>**Cambiar estado:** Desde los detalles del pedido, usa el desplegable "Estado" para actualizarlo (ej., de "Procesando" a "Completado" una vez enviado).</li>
                        <li>**Añadir notas:** Puedes añadir notas privadas o para el cliente en la sección de "Notas del pedido".</li>
                        <li>**Reembolsos:** Si necesitas hacer un reembolso, puedes gestionarlo directamente desde los detalles del pedido.</li>
                    </ol>
                    <p class="tip">**Consejo:** Filtra los pedidos por estado para una gestión más eficiente, por ejemplo, para ver solo los pedidos "Procesando" que necesitan ser enviados.</p>
                </div>
            </div>

            <div class="woo-card" data-card="productos">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-products"></span> Productos</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Añade, edita y organiza todos los productos de tu tienda.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('edit.php?post_type=product') ); ?>" class="button button-primary">Ver Productos</a>
                    <a href="<?php echo esc_url( admin_url('post-new.php?post_type=product') ); ?>" class="button button-secondary">Añadir Nuevo</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Cómo Gestionar Productos</h3>
                    <p>Aquí controlas todo lo relacionado con tu catálogo de productos. Es fundamental mantener esta sección actualizada.</p>
                    <ol>
                        <li>**Añadir producto:** Haz clic en "Añadir Nuevo" para crear un producto desde cero. Rellena el nombre, descripción, precio, inventario, imágenes y categorías.</li>
                        <li>**Editar producto:** Haz clic en el nombre de un producto existente para modificar sus detalles, precios, stock, etc.</li>
                        <li>**Categorías y etiquetas:** Organiza tus productos usando categorías y etiquetas para facilitar la navegación a tus clientes.</li>
                        <li>**Imágenes:** Asegúrate de que cada producto tenga imágenes de alta calidad para atraer a los compradores.</li>
                    </ol>
                    <p class="tip">**Consejo:** Utiliza productos simples para artículos únicos y productos variables para artículos con opciones (tallas, colores).</p>
                </div>
            </div>
            
            <div class="woo-card" data-card="categorias">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-category"></span> Categorías</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Gestiona las categorías de tus productos para una mejor organización.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('edit-tags.php?taxonomy=product_cat&post_type=product') ); ?>" class="button button-primary">Ver Categorías</a>
                    <a href="<?php echo esc_url( admin_url('edit-tags.php?taxonomy=product_cat&post_type=product') ); ?>" class="button button-secondary">Añadir Categoría</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Cómo Gestionar Categorías</h3>
                    <p>Las categorías te permiten agrupar productos similares, lo que mejora la navegación para tus clientes y la gestión de tu inventario.</p>
                    <ol>
                        <li>**Crear Categoría:** Añade un nuevo nombre, slug y descripción para organizar tus productos. Puedes crear subcategorías.</li>
                        <li>**Asignar a Productos:** Al editar un producto, puedes asignarlo a una o varias categorías existentes.</li>
                        <li>**Imágenes de Categoría:** Algunas plantillas de tienda permiten asignar una imagen destacada a cada categoría, lo que mejora la estética de tu tienda.</li>
                    </ol>
                    <p class="tip">**Consejo:** Una buena estructura de categorías ayuda a tus clientes a encontrar lo que buscan rápidamente y mejora el SEO de tu tienda.</p>
                </div>
            </div>

            <div class="woo-card" data-card="clientes">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-admin-users"></span> Clientes</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Visualiza y gestiona la información de tus clientes.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('users.php') ); ?>" class="button button-primary">Ver Clientes</a>
                    <a href="<?php echo esc_url( admin_url('admin.php?page=wc-customers') ); ?>" class="button button-secondary">Informes de Clientes</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Cómo Gestionar Clientes</h3>
                    <p>La sección de clientes te permite ver quiénes compran en tu tienda y sus datos de contacto. Puedes acceder a su historial de pedidos.</p>
                    <ol>
                        <li>**Ver perfil:** Haz clic en el nombre de un cliente para ver su perfil de usuario, incluyendo dirección de envío y facturación.</li>
                        <li>**Historial de pedidos:** Dentro del perfil del cliente, puedes ver todos los pedidos que ha realizado.</li>
                        <li>**Informes:** Los informes de clientes te darán una visión general de tus compradores más activos o los que más gastan.</li>
                    </ol>
                    <p class="tip">**Consejo:** Mantener los datos de los clientes actualizados puede ser útil para campañas de marketing o para la resolución de problemas.</p>
                </div>
            </div>

            <div class="woo-card" data-card="estadisticas">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-chart-bar"></span> Estadísticas</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Analiza el rendimiento de tu tienda con estadísticas detalladas.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('admin.php?page=wc-admin&path=/analytics/overview') ); ?>" class="button button-primary">Ver Estadísticas</a>
                    <a href="<?php echo esc_url( admin_url('admin.php?page=wc-settings&tab=advanced') ); ?>" class="button button-secondary">Ajustes Avanzados</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Cómo Utilizar las Estadísticas</h3>
                    <p>Las estadísticas de WooCommerce son una herramienta poderosa para entender el rendimiento de tu tienda. Puedes analizar ventas, productos más populares y el comportamiento del cliente.</p>
                    <ol>
                        <li>**Visión general:** Obtén un resumen rápido de las ventas, pedidos y ganancias.</li>
                        <li>**Estadísticas de ventas:** Desglosa las ventas por fecha, producto, categoría o cupón.</li>
                        <li>**Estadísticas de productos:** Descubre qué productos se venden mejor y cuáles no.</li>
                        <li>**Estadísticas de ingresos:** Analiza las tendencias de ingresos a lo largo del tiempo.</li>
                    </ol>
                    <p class="tip">**Consejo:** Revisa tus estadísticas regularmente para identificar tendencias, optimizar tus productos y mejorar tus estrategias de marketing.</p>
                </div>
            </div>

            <div class="woo-card" data-card="cupones">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-tag"></span> Cupones</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Crea y gestiona códigos de cupón para promociones y descuentos.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('edit.php?post_type=shop_coupon') ); ?>" class="button button-primary">Ver Cupones</a>
                    <a href="<?php echo esc_url( admin_url('post-new.php?post_type=shop_coupon') ); ?>" class="button button-secondary">Añadir Nuevo Cupón</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Cómo Crear y Gestionar Cupones</h3>
                    <p>Los cupones son una excelente herramienta para ofrecer descuentos, atraer nuevos clientes y recompensar a los existentes. Puedes configurar diferentes tipos de descuentos.</p>
                    <ol>
                        <li>**Tipo de descuento:** Elige entre porcentaje de descuento, descuento fijo de carrito o descuento fijo de producto.</li>
                        <li>**Restricciones:** Define si el cupón aplica a productos específicos, categorías, o si hay un importe mínimo de compra.</li>
                        <li>**Fecha de caducidad:** Establece un límite de tiempo para tus promociones.</li>
                        <li>**Límites de uso:** Controla cuántas veces se puede usar un cupón en total o por cliente.</li>
                    </ol>
                    <p class="tip">**Consejo:** Los cupones son ideales para campañas de email marketing o para promociones en redes sociales.</p>
                </div>
            </div>

            <div class="woo-card" data-card="ajustes">
                <div class="card-header">
                    <h2><span class="dashicons dashicons-admin-settings"></span> Ajustes</h2>
                    <span class="info-icon dashicons dashicons-info" title="Haz clic para obtener más información"></span>
                </div>
                <p>Configura los ajustes generales de tu tienda WooCommerce.</p>
                <div class="card-actions">
                    <a href="<?php echo esc_url( admin_url('admin.php?page=wc-settings') ); ?>" class="button button-primary">Ir a Ajustes</a>
                </div>
                <div class="info-content hidden">
                    <button class="close-button dashicons dashicons-no-alt"></button>
                    <h3>Ajustes Clave de WooCommerce</h3>
                    <p>Esta sección es el corazón de la configuración de tu tienda. Aquí puedes configurar impuestos, envíos, métodos de pago, correos electrónicos y más.</p>
                    <ol>
                        <li>**General:** Moneda, ubicación de la tienda, opciones de venta.</li>
                        <li>**Productos:** Unidades de peso/dimensión, inventario, productos descargables.</li>
                        <li>**Impuestos:** Configuración de tasas impositivas y cómo se muestran los precios.</li>
                        <li>**Envío:** Zonas de envío, métodos y clases de envío.</li>
                        <li>**Pagos:** Configuración de pasarelas de pago como Stripe, PayPal, Transferencia bancaria, etc.</li>
                        <li>**Cuentas y privacidad:** Políticas de privacidad, registro de clientes.</li>
                        <li>**Correos electrónicos:** Plantillas y configuración de emails transaccionales.</li>
                    </ol>
                    <p class="tip">**Consejo:** Revisa los ajustes de envío y pago con regularidad, especialmente si cambian las tarifas o proveedores.</p>
                </div>
            </div>

        </div>
    </div>

    <style>
        /* Estilos generales del contenedor */
        .custom-woo-dashboard-wrap {
            max-width: 1300px;
            margin: 20px auto;
            padding: 20px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
            color: #333;
        }

        .custom-woo-dashboard-wrap h1 {
            color: #555;
            font-size: 2.2em;
            margin-top: 0;
            margin-bottom: 5px;
            border-bottom: 2px solid #eee;
            padding-bottom: 10px;
        }

        .custom-woo-dashboard-wrap .subtitle {
            font-size: 1.1em;
            color: #777;
            margin-bottom: 30px;
        }

        .woo-cards-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
            margin-top: 30px;
        }

        .woo-card {
            background: #f9f9f9;
            border: 1px solid #e0e0e0;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.03);
            display: flex;
            flex-direction: column;
            justify-content: space-between;
            transition: all 0.2s ease-in-out;
            min-height: 220px;
            position: relative;
        }

        .woo-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 6px 15px rgba(0,0,0,0.08);
            border-color: #d0d0d0;
        }

        .woo-card .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            border-bottom: 1px solid #e9e9e9;
            padding-bottom: 10px;
        }

        .woo-card .card-header h2 {
            font-size: 1.6em;
            color: #0073aa;
            margin: 0;
            display: flex;
            align-items: center;
        }

        .woo-card .card-header h2 .dashicons {
            margin-right: 10px;
            font-size: 1.2em;
        }

        .woo-card .info-icon {
            font-size: 1.5em;
            color: #0073aa;
            cursor: pointer;
            transition: color 0.2s ease;
            position: absolute;
            top: 20px;
            right: 20px;
        }

        .woo-card .info-icon:hover {
            color: #005177;
        }

        .woo-card p {
            font-size: 0.95em;
            color: #666;
            line-height: 1.5;
            margin-bottom: 20px;
        }

        .woo-card .card-actions {
            margin-top: auto;
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }

        .woo-card .button {
            padding: 8px 15px;
            font-size: 0.9em;
            border-radius: 5px;
            text-decoration: none;
            text-align: center;
            flex-grow: 1;
            display: inline-block;
        }

        .woo-card .button-primary {
            background-color: #0073aa;
            color: #fff;
            border-color: #006799;
            box-shadow: 0 1px 0 #006799;
        }

        .woo-card .button-primary:hover {
            background-color: #005177;
            border-color: #005177;
            color: #fff;
        }

        .woo-card .button-secondary {
            background-color: #f3f3f3;
            color: #555;
            border-color: #bbb;
            box-shadow: 0 1px 0 #bbb;
        }

        .woo-card .button-secondary:hover {
            background-color: #e3e3e3;
            border-color: #888;
            color: #333;
        }

        /* Contenido de información (oculto por defecto) */
        .woo-card .info-content {
            border-top: 1px dashed #ddd;
            padding-top: 15px;
            margin-top: 20px;
            font-size: 0.9em;
            color: #444;
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.4s ease-out, padding-top 0.4s ease-out;
            opacity: 0;
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.98);
            border-radius: 10px;
            padding: 25px;
            box-sizing: border-box;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
            z-index: 10;
            transform: scale(0.95);
            transform-origin: center;
            transition: all 0.3s ease-in-out;
            display: flex;
            flex-direction: column;
        }

        .woo-card .info-content.active {
            max-height: 100%;
            opacity: 1;
            transform: scale(1);
            overflow-y: auto;
        }
        
        /* Esta es la línea crucial para el CSS. */
        /* Aseguramos que se oculte el wrapper del panel de actividad de WooCommerce Admin */
        .woocommerce-layout__activity-panel-wrapper {
            display: none !important;
        }

        /* Botón de cierre para el info-content */
        .info-content .close-button {
            position: absolute;
            top: 10px;
            right: 10px;
            font-size: 1.5em;
            line-height: 1;
            color: #888;
            cursor: pointer;
            background: none;
            border: none;
            padding: 5px;
            border-radius: 50%;
            transition: color 0.2s ease, background-color 0.2s ease;
            text-decoration: none;
            display: block;
        }

        .info-content .close-button:hover {
            color: #333;
            background-color: #f0f0f0;
        }

        /* Estilos para la página de ajustes de usuarios (NUEVOS ESTILOS) */
        .woo-user-settings-wrap {
            max-width: 900px; /* Ajusta el ancho para la nueva interfaz */
            margin: 20px auto;
            padding: 20px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
            color: #333;
        }
        .woo-user-settings-wrap h1 {
            color: #555;
            font-size: 2.2em;
            margin-top: 0;
            margin-bottom: 5px;
            border-bottom: 2px solid #eee;
            padding-bottom: 10px;
        }
        .woo-user-settings-wrap p.description {
            font-style: italic;
            color: #777;
            margin-top: -10px;
            margin-bottom: 25px;
        }
        .woo-user-settings-wrap .user-selection-area {
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 20px;
            background-color: #fdfdfd;
        }
	.user-list {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-top: 30px;
}

.user-card {
    background-color: #ffffff;
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 15px 20px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.user-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
}

.user-card input[type="checkbox"] {
    margin-right: 12px;
    transform: scale(1.2);
}

.user-card label {
    display: flex;
    align-items: center;
    cursor: pointer;
}

.user-info {
    display: flex;
    flex-direction: column;
    font-size: 14px;
    color: #333;
}

.user-info strong {
    font-size: 15px;
    color: #0073aa;
}

.user-email {
    font-size: 13px;
    color: #666;
}

        .woo-user-settings-wrap .search-box {
            width: 100%;
            padding: 10px;
            margin-bottom: 15px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
            font-size: 1em;
        }
        .woo-user-settings-wrap .user-list {
            max-height: 400px; /* Altura máxima para la lista de usuarios */
            overflow-y: auto; /* Scroll si hay muchos usuarios */
            border: 1px solid #eee;
            background-color: #fff;
            padding: 10px;
            border-radius: 4px;
        }
        .woo-user-settings-wrap .user-list-item {
            display: flex;
            align-items: center;
            padding: 8px 10px;
            border-bottom: 1px solid #f0f0f0;
            transition: background-color 0.2s ease;
        }
        .woo-user-settings-wrap .user-list-item:last-child {
            border-bottom: none;
        }
        .woo-user-settings-wrap .user-list-item:hover {
            background-color: #f5f5f5;
        }
        .woo-user-settings-wrap .user-list-item input[type="checkbox"] {
            margin-right: 10px;
            transform: scale(1.2); /* Checkbox un poco más grande */
        }
        .woo-user-settings-wrap .user-list-item label {
            margin: 0;
            font-weight: normal;
            cursor: pointer;
            display: flex;
            align-items: center;
            width: 100%; /* Para que el clic en la etiqueta también funcione */
        }
        .woo-user-settings-wrap .user-list-item span.user-email {
            font-size: 0.9em;
            color: #888;
            margin-left: auto; /* Empuja el email a la derecha */
        }
        .woo-user-settings-wrap .button-primary {
            margin-top: 25px;
        }


        /* Responsive adjustments */
        @media (max-width: 991px) {
            .woo-cards-container {
                grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
                gap: 20px;
            }
            .custom-woo-dashboard-wrap {
                padding: 15px;
            }
            .custom-woo-dashboard-wrap h1 {
                font-size: 1.8em;
            }
            .woo-card .card-header h2 {
                font-size: 1.4em;
            }
            .woo-card p {
                font-size: 0.9em;
            }
        }

        @media (max-width: 767px) {
            .woo-cards-container {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            .custom-woo-dashboard-wrap {
                padding: 10px;
            }
            .custom-woo-dashboard-wrap h1 {
                font-size: 1.6em;
            }
            .woo-card {
                padding: 20px;
                min-height: auto;
            }
            .woo-card .info-icon {
                font-size: 1.3em;
                top: 15px;
                right: 15px;
            }
            .woo-card .button {
                flex-basis: 100%;
            }
            .woo-user-settings-wrap {
                padding: 15px;
            }
            .woo-user-settings-wrap .user-list {
                max-height: 300px;
            }
        }
    </style>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Script para las tarjetas de información
            const infoIcons = document.querySelectorAll('.woo-card .info-icon');

            infoIcons.forEach(icon => {
                icon.addEventListener('click', function() {
                    const card = this.closest('.woo-card');
                    const infoContent = card.querySelector('.info-content');
                    
                    infoContent.classList.add('active'); // Muestra el contenido
                });
            });

            document.querySelectorAll('.info-content .close-button').forEach(button => {
                button.addEventListener('click', function() {
                    this.closest('.info-content').classList.remove('active');
                });
            });

            // --- Script para la página de Ajustes de Usuarios ---
            const searchInput = document.getElementById('user-search-input');
            const userList = document.getElementById('user-checkbox-list');

            if (searchInput && userList) {
                searchInput.addEventListener('keyup', function() {
                    const searchTerm = this.value.toLowerCase();
                    const listItems = userList.querySelectorAll('.user-list-item');

                    listItems.forEach(item => {
                        const userName = item.querySelector('label').textContent.toLowerCase();
                        if (userName.includes(searchTerm)) {
                            item.style.display = 'flex'; // Mostrar el elemento
                        } else {
                            item.style.display = 'none'; // Ocultar el elemento
                        }
                    });
                });
            }
        });
    </script>
    <?php
}

// 3. Función para mostrar el contenido de la página de ajustes de usuarios
function display_woo_dashboard_user_settings() {
    // Solo permitir que administradores accedan a esta página
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( 'No tienes permisos suficientes para acceder a esta página.', 'textdomain' ) );
    }

    // Procesar el guardado de la configuración
    if ( isset( $_POST['submit_woo_dashboard_users'] ) ) {
        check_admin_referer( 'woo_dashboard_user_settings_nonce' ); // Nonce de seguridad

        // Recopilar los IDs de los checkboxes marcados
        $selected_users = isset( $_POST['woo_dashboard_affected_users'] ) ? array_map( 'intval', $_POST['woo_dashboard_affected_users'] ) : array();
        update_option( WOO_CUSTOM_DASHBOARD_USERS_OPTION, $selected_users );
        ?>
        <div class="notice notice-success is-dismissible">
            <p><strong><?php _e( 'Ajustes guardados correctamente.', 'textdomain' ); ?></strong></p>
        </div>
        <?php
    }

    // Obtener los usuarios actualmente afectados
    $affected_user_ids = get_option( WOO_CUSTOM_DASHBOARD_USERS_OPTION, array() );

    // Obtener todos los usuarios con capacidad 'edit_products' (o similar, para filtrar)
    $args = array(
        'fields' => array( 'ID', 'display_name', 'user_email' ), // Ahora también el email
        'orderby' => 'display_name',
        'order' => 'ASC',
        'role__in' => array( 'administrator', 'editor', 'shop_manager', 'customer' ), // Incluye roles que puedan ser afectados
        // Puedes ajustar los roles aquí según quiénes pueden ser "usuarios del plugin"
    );
    $all_users = get_users( $args );
    ?>
    <div class="wrap woo-user-settings-wrap">
        <h1>Ajustes de Usuarios para el Centro de Control WooCommerce</h1>
        <p class="description">Selecciona los usuarios que tendrán el acceso restringido al Centro de Control WooCommerce y verán el menú lateral y las funciones de administración estándar ocultas.</p>
        <form method="post" action="">
            <?php wp_nonce_field( 'woo_dashboard_user_settings_nonce' ); ?>

            <div class="user-selection-area">
               
                
              <div class="user-list" id="user-checkbox-list">
    <?php
    if ( ! empty( $all_users ) ) {
        foreach ( $all_users as $user ) {
            $checked = in_array( $user->ID, $affected_user_ids ) ? 'checked' : '';
            echo '<div class="user-card">';
            echo '<label for="user_' . esc_attr( $user->ID ) . '">';
            echo '<input type="checkbox" name="woo_dashboard_affected_users[]" id="user_' . esc_attr( $user->ID ) . '" value="' . esc_attr( $user->ID ) . '" ' . $checked . '>';
            echo '<div class="user-info">';
            echo '<strong>' . esc_html( $user->display_name ) . '</strong><br>';
            echo '<span class="user-email">' . esc_html( $user->user_email ) . '</span>';
            echo '</div>';
            echo '</label>';
            echo '</div>';
        }
    } else {
        echo '<p>No se encontraron usuarios con los roles especificados.</p>';
    }
    ?>
</div>
            </div>

            <input type="submit" name="submit_woo_dashboard_users" class="button button-primary" value="Guardar Cambios">
        </form>
    </div>
    <?php
}


// Función auxiliar para comprobar si el usuario actual está afectado por el plugin
function is_user_affected_by_woo_dashboard() {
    $current_user = wp_get_current_user();
    if ( ! $current_user || ! $current_user->exists() ) {
        return false;
    }

    $affected_user_ids = get_option( WOO_CUSTOM_DASHBOARD_USERS_OPTION, array() );

    // Si la opción está vacía o no es un array, no hay usuarios afectados
    if ( ! is_array( $affected_user_ids ) || empty( $affected_user_ids ) ) {
        return false;
    }

    return in_array( $current_user->ID, $affected_user_ids );
}


// 4. Redirigir a los usuarios seleccionados a esta nueva página al iniciar sesión en el backend (solo en escritorio)
function redirect_specific_woo_user_to_custom_dashboard() {
    // Solo si estamos en el área de administración y no es una petición AJAX
    if ( is_admin() && ! wp_doing_ajax() ) {
        
        if ( is_user_affected_by_woo_dashboard() ) {

            // Detección básica de escritorio.
            $is_desktop = true;
            $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
            if ( preg_match('/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|rim)|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i', $user_agent) ||
                 preg_match('/(tablet|ipad|playbook|silk)|(android(?!.*mobile))/i', $user_agent) ) {
                $is_desktop = false;
            }

            $current_request_uri = $_SERVER['REQUEST_URI'];
            $custom_dashboard_target_url = admin_url( 'admin.php?page=' . CUSTOM_WOO_DASHBOARD_SLUG );

            // Redirigir solo si es de escritorio Y la página actual es 'index.php' (el dashboard principal)
            // O si la página actual es 'admin.php' SIN el slug de tu dashboard personalizado.
            // Esto evita que la redirección se active cuando ya estás navegando a otras secciones.
            if ( $is_desktop && 
                 ( basename($current_request_uri) === 'index.php' || 
                   ( basename($current_request_uri) === 'admin.php' && !isset($_GET['page']) ) ||
                   ( strpos($current_request_uri, 'admin.php') !== false && isset($_GET['page']) && $_GET['page'] === 'dashboard' ) // Para algunos casos específicos de dashboard
                 )
            ) {
                // Aseguramos que solo redirija si el destino no es ya la página actual (evitar bucle)
                if ( ! ( strpos($current_request_uri, 'page=' . CUSTOM_WOO_DASHBOARD_SLUG) !== false ) ) {
                    wp_redirect( $custom_dashboard_target_url );
                    exit;
                }
            }
        }
    }
}
add_action( 'admin_init', 'redirect_specific_woo_user_to_custom_dashboard' );


// Ocultar el menú lateral completo de WordPress para los usuarios afectados
function hide_admin_menu_for_specific_woo_user_with_css() {
    if ( is_admin() && is_user_affected_by_woo_dashboard() ) {
        ?>
        <style type="text/css">
            /* Oculta la barra lateral de navegación */
            #adminmenumain, #adminmenuwrap {
                display: none !important;
            }
            /* Ajusta el contenido principal para que ocupe todo el ancho */
            #wpcontent, #wpfooter {
                margin-left: 0 !important;
            }
            /* Si tienes la barra lateral activa en móvil, asegúrate de que también se oculte */
            @media screen and (max-width: 782px) {
                #adminmenumain, #adminmenuwrap {
                    display: none !important;
                }
                #wpcontent, #wpfooter {
                    margin-left: 0 !important;
                }
            }
        </style>
        <?php
    }
}
add_action('admin_head', 'hide_admin_menu_for_specific_woo_user_with_css');


// **DESHABILITAR EL PANEL DE ACTIVIDAD Y OTRAS CARACTERÍSTICAS DE WOOCOMMERCE ADMIN** para los usuarios afectados
add_filter( 'woocommerce_admin_features', 'deshabilitar_panel_actividad_woocommerce' );

function deshabilitar_panel_actividad_woocommerce( $features ) {
    if ( is_user_affected_by_woo_dashboard() ) {
        $features = array_diff( $features, [
            'activity-panel', // Para deshabilitar el panel de actividad
            'onboarding',     // Desactiva el asistente de configuración inicial
            'setup-wizard',   // Desactiva el asistente de configuración si reaparece
            'homepage',       // Desactiva la nueva página de inicio de WooCommerce Admin
            'dashboard',      // Desactiva el dashboard de WooCommerce Admin (no el de WP)
            'task-lists',     // Desactiva las listas de tareas
            // 'navigation',  // Comentado: si ocultas el menú lateral, esto es menos crítico. Si lo activas, podría afectar tu navegación personalizada.
            'analytics',      // Si usas tu propio card de "Estadísticas", esto puede ser redundante.
            'marketing'       // Desactiva la sección de marketing de WooCommerce Admin
        ] );
    }
    return $features;
}

// Ocultar el Dashboard original del menú para este usuario (PHP)
function remove_dashboard_menu_for_specific_woo_user() {
    if ( is_user_affected_by_woo_dashboard() ) {
        remove_menu_page('index.php'); // Oculta el elemento "Escritorio" principal
    }
}
add_action('admin_menu', 'remove_dashboard_menu_for_specific_woo_user', 999);

// **ELIMINAR WIDGETS DE ESCRITORIO DE WOOCOMMERCE** para los usuarios afectados
function remove_woocommerce_dashboard_widgets_for_specific_user() {
    if ( is_user_affected_by_woo_dashboard() ) {
        remove_meta_box( 'woocommerce_dashboard_status', 'dashboard', 'normal' );
        remove_meta_box( 'woocommerce_dashboard_recent_orders', 'dashboard', 'normal' );
        remove_meta_box( 'woocommerce_dashboard_product_activity', 'dashboard', 'normal' );
        remove_meta_box( 'woocommerce_dashboard_activity', 'dashboard', 'normal' );
        // Añadir otros widgets que puedan aparecer y molestar
        remove_meta_box( 'dashboard_activity', 'dashboard', 'normal' ); // Widget de actividad general de WP
        remove_meta_box( 'dashboard_right_now', 'dashboard', 'normal' ); // Resumen "De un vistazo"
        remove_meta_box( 'dashboard_primary', 'dashboard', 'normal' ); // Novedades y eventos de WordPress
        remove_meta_box( 'dashboard_quick_press', 'dashboard', 'side' ); // Publicación rápida
        remove_meta_box( 'dashboard_recent_drafts', 'dashboard', 'side' ); // Borradores recientes
        remove_meta_box( 'dashboard_incoming_links', 'dashboard', 'normal' ); // Enlaces entrantes (antiguo)
        remove_meta_box( 'dashboard_plugins', 'dashboard', 'normal' ); // Plugins (a veces añadido por plugins de seguridad)
    }
}
add_action( 'wp_dashboard_setup', 'remove_woocommerce_dashboard_widgets_for_specific_user' );

// **ASEGURARSE DE QUE NO SE CARGUEN LOS SCRIPTS Y ESTILOS DEL PANEL DE ACTIVIDAD Y OTROS COMPONENTES DE WOOCOMMERCE ADMIN** para los usuarios afectados
function remove_woocommerce_admin_activity_panel_assets_for_specific_user() {
    if ( is_user_affected_by_woo_dashboard() ) {
        // IDs de los scripts y estilos conocidos del panel de actividad y la interfaz de WooCommerce Admin
        $scripts_to_dequeue = [
            'wc-admin-app',
            'wc-components',
            'wc-csv',
            'wc-date',
            'wc-navigation',
            'wc-number',
            'wc-currency',
            'wc-order',
            'wc-settings',
            'wc-customer',
            'wc-product',
            'wc-report', // Incluye informes/estadísticas antiguas de WC
            'wc-tracks',
            'wc-setup',
            'wc-payment-gateways',
            'wc-shipping-zones',
            'wc-shipping-classes',
            'wc-tax-rates',
            'wc-coupons',
            'wc-activity-panel',
            'wc-admin-data',
            'wc-admin-dashboard',
            'wc-admin-homepage',
            'wc-onboarding',
            'wc-onboarding-tasks',
            'woocommerce-admin-app',
            // Añadidos o confirmados para ser más exhaustivos
            'selectWoo', // Es un script de Select2 usado por WC Admin
            'wc-enhanced-select', // Dependencia de Select2 para WC
            'wc-block-components',
            'wc-block-editor',
            'wc-admin-settings',
            'wc-admin-analytics', // Scripts para las analíticas/informes nuevos
            'wc-admin-reports',   // Scripts para las analíticas/informes nuevos
        ];

        foreach ( $scripts_to_dequeue as $handle ) {
            wp_dequeue_script( $handle );
            wp_deregister_script( $handle );
        }

        $styles_to_dequeue = [
            'wc-admin-app',
            'wc-components',
            'wc-activity-panel',
            'woocommerce_admin_styles',
            'wc-blocks-vendors-style',
            'wc-blocks-style',
            'wc-admin-dashboard',
            'wc-admin-homepage',
            // Añadidos o confirmados para ser más exhaustivos
            'select2', // Estilos para Select2
            'woocommerce_admin_menu_styles',
            'wc-admin-settings',
            'wc-admin-analytics',
            'wc-admin-reports',
        ];

        foreach ( $styles_to_dequeue as $handle ) {
            wp_dequeue_style( $handle );
            wp_deregister_style( $handle );
        }
    }
}
add_action( 'admin_enqueue_scripts', 'remove_woocommerce_admin_activity_panel_assets_for_specific_user', 9999 );

// **OCULTAR OTROS ELEMENTOS DE LA BARRA DE ADMINISTRACIÓN SUPERIOR PARA LOS USUARIOS AFECTADOS**
function hide_admin_bar_items_for_specific_woo_user( $wp_admin_bar ) {
    if ( is_user_affected_by_woo_dashboard() ) {
        // Eliminar el logo de WordPress
        $wp_admin_bar->remove_node('wp-logo');
        // Eliminar el nodo de "Acerca de WordPress"
        $wp_admin_bar->remove_node('about');
        // Eliminar el nodo de "WordPress.org"
        $wp_admin_bar->remove_node('wporg');
        // Eliminar el nodo de "Documentación"
        $wp_admin_bar->remove_node('documentation');
        // Eliminar el nodo de "Foros de soporte"
        $wp_admin_bar->remove_node('support-forums');
        // Eliminar el nodo de "Comentarios"
        $wp_admin_bar->remove_node('feedback');
        // Eliminar el nodo de "Novedades" (si existe)
        $wp_admin_bar->remove_node('updates');
        // Eliminar el nodo de "Comentarios"
        $wp_admin_bar->remove_node('comments');
        // Eliminar el nodo de "Nuevo contenido" (publicación rápida)
        $wp_admin_bar->remove_node('new-content');
        // Eliminar el nodo de "Visitar sitio" (si no quieres que salgan del admin)
        // $wp_admin_bar->remove_node('view-site');
        // Eliminar el nodo de "WooCommerce" general (si aparece y no quieres que accedan por ahí)
        $wp_admin_bar->remove_node('woocommerce');
        // Eliminar el nodo de "Personalizar" (en frontend si tienen acceso)
        $wp_admin_bar->remove_node('customize');
        // Eliminar el nodo de "WPForms" o similar si hay plugins que añaden nodos
        $wp_admin_bar->remove_node('wpforms');
        // Eliminar el nodo de "Yoast SEO" o similar
        $wp_admin_bar->remove_node('wpseo-menu');
    }
}
add_action( 'admin_bar_menu', 'hide_admin_bar_items_for_specific_woo_user', 999 );

// **NUEVA FUNCIÓN: Añadir botón flotante para volver al Centro WooCommerce** para los usuarios afectados
function add_floating_back_to_dashboard_button() {
    // Solo si es el usuario objetivo y NO estamos ya en el dashboard personalizado
    if ( is_user_affected_by_woo_dashboard() && 
         strpos($_SERVER['REQUEST_URI'], 'page=' . CUSTOM_WOO_DASHBOARD_SLUG) === false ) {
        ?>
        <a href="<?php echo esc_url( admin_url('admin.php?page=' . CUSTOM_WOO_DASHBOARD_SLUG) ); ?>" 
           id="woo-back-to-dashboard-button" 
           title="Volver al Centro WooCommerce">
            <span class="dashicons dashicons-arrow-left-alt2"></span>
            Volver al Centro
        </a>
        <style type="text/css">
            #woo-back-to-dashboard-button {
                position: fixed;
                bottom: 70px; /* ¡Cambiado de 20px a 70px! */
                right: 20px;
                background-color: #0073aa; /* Color primario de WP Admin */
                color: #fff;
                padding: 12px 18px;
                border-radius: 8px;
                box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
                text-decoration: none;
                font-size: 1em;
                font-weight: 600;
                display: flex;
                align-items: center;
                gap: 8px;
                z-index: 10000; /* Asegura que esté por encima de otros elementos */
                transition: background-color 0.2s ease, transform 0.2s ease;
            }

            #woo-back-to-dashboard-button:hover {
                background-color: #005177; /* Tono más oscuro al pasar el ratón */
                transform: translateY(-2px);
                color: #fff; /* Asegura que el color del texto no cambie en hover */
            }

            #woo-back-to-dashboard-button .dashicons {
                font-size: 1.4em;
                line-height: 1;
            }

            /* Opcional: Ocultar en móviles si el espacio es un problema */
            @media (max-width: 768px) {
                #woo-back-to-dashboard-button {
                    bottom: 65px; /* También ajustado para móviles (15px + 50px) */
                    right: 15px;
                    padding: 10px 15px;
                    font-size: 0.9em;
                    gap: 5px;
                }
                #woo-back-to-dashboard-button span.dashicons {
                    /* Solo mostrar el icono en pantallas más pequeñas para ahorrar espacio */
                    margin-right: 0;
                }
                /* Si solo quieres el icono en móvil: */
                #woo-back-to-dashboard-button {
                    width: 50px;
                    height: 50px;
                    border-radius: 50%;
                    text-indent: -9999px; /* Oculta el texto */
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    padding: 0;
                }
                #woo-back-to-dashboard-button .dashicons {
                    text-indent: 0; /* Muestra el icono */
                }
            }
        </style>
        <?php
    }
}
add_action('admin_footer', 'add_floating_back_to_dashboard_button');